/*
 * Decompiled with CFR 0.152.
 */
package us.fatehi.utility.datasource;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Deque;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import us.fatehi.utility.SQLRuntimeException;
import us.fatehi.utility.Utility;
import us.fatehi.utility.database.DatabaseUtility;
import us.fatehi.utility.datasource.AbstractDatabaseConnectionSource;
import us.fatehi.utility.datasource.PooledConnectionUtility;
import us.fatehi.utility.datasource.UserCredentials;
import us.fatehi.utility.string.StringFormat;

final class SimpleDatabaseConnectionSource
extends AbstractDatabaseConnectionSource {
    private static final Logger LOGGER = Logger.getLogger(SimpleDatabaseConnectionSource.class.getName());
    private final String connectionUrl;
    private final Properties jdbcConnectionProperties;
    private final Deque<Connection> connectionPool;
    private final Deque<Connection> usedConnections;

    SimpleDatabaseConnectionSource(String connectionUrl, Map<String, String> connectionProperties, UserCredentials userCredentials, Consumer<Connection> connectionInitializer) {
        super(connectionInitializer);
        this.connectionUrl = Utility.requireNotBlank(connectionUrl, "No database connection URL provided");
        Objects.requireNonNull(userCredentials, "No user credentials provided");
        String user = userCredentials.user();
        String password = userCredentials.password();
        if (Utility.isBlank(user)) {
            LOGGER.log(Level.WARNING, "Database user is not provided");
        }
        if (Utility.isBlank(password)) {
            LOGGER.log(Level.WARNING, "Database password is not provided");
        }
        this.jdbcConnectionProperties = SimpleDatabaseConnectionSource.createConnectionProperties(connectionUrl, connectionProperties, user, password);
        this.connectionPool = new LinkedBlockingDeque<Connection>();
        this.usedConnections = new LinkedBlockingDeque<Connection>();
    }

    @Override
    public void close() throws Exception {
        ArrayList<Connection> connections = new ArrayList<Connection>();
        connections.addAll(this.connectionPool);
        connections.addAll(this.usedConnections);
        for (Connection connection : connections) {
            try {
                connection.close();
                LOGGER.log(Level.INFO, new StringFormat("Closed database connection <%s>", connection));
            }
            catch (Exception e) {
                LOGGER.log(Level.WARNING, "Cannot close connection", e);
            }
        }
        if (!this.usedConnections.isEmpty()) {
            LOGGER.log(Level.SEVERE, "Abnormal termination - not all database connections are closed");
        }
        this.connectionPool.clear();
        this.usedConnections.clear();
    }

    @Override
    public synchronized Connection get() {
        Connection connection;
        if (this.connectionPool.isEmpty()) {
            connection = SimpleDatabaseConnectionSource.getConnection(this.connectionUrl, this.jdbcConnectionProperties);
            this.connectionPool.add(connection);
        }
        connection = this.connectionPool.removeFirst();
        this.usedConnections.add(connection);
        this.connectionInitializer.accept(connection);
        LOGGER.log(Level.FINE, new StringFormat("Initialized database connection <%s> with <%s>", connection, this.connectionInitializer));
        return PooledConnectionUtility.newPooledConnection(connection, this);
    }

    @Override
    public synchronized boolean releaseConnection(Connection connection) {
        boolean removed = this.usedConnections.remove(connection);
        try {
            Connection unwrappedConnection = connection.unwrap(Connection.class);
            DatabaseUtility.checkConnection(unwrappedConnection);
        }
        catch (SQLException e) {
            LOGGER.log(Level.WARNING, "Cannot check connection before returning to the pool - " + e.getMessage());
            LOGGER.log(Level.FINE, "Cannot check connection before returning to the pool - ", e);
        }
        this.connectionPool.add(connection);
        return removed;
    }

    protected void finalize() throws Throwable {
        if (!this.connectionPool.isEmpty() || !this.usedConnections.isEmpty()) {
            throw new SQLRuntimeException("Connection pool is not closed");
        }
        super.finalize();
    }
}

